/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.tools;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.MultiObjectDeleteException;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.s3a.Invoker;
import org.apache.hadoop.fs.s3a.S3AFileStatus;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3a.S3ALocatedFileStatus;
import org.apache.hadoop.fs.s3a.UnknownStoreException;
import org.apache.hadoop.fs.s3a.impl.DirMarkerTracker;
import org.apache.hadoop.fs.s3a.impl.DirectoryPolicy;
import org.apache.hadoop.fs.s3a.impl.DirectoryPolicyImpl;
import org.apache.hadoop.fs.s3a.impl.StoreContext;
import org.apache.hadoop.fs.s3a.s3guard.S3GuardTool;
import org.apache.hadoop.fs.s3a.tools.MarkerToolOperations;
import org.apache.hadoop.fs.shell.CommandFormat;
import org.apache.hadoop.fs.statistics.IOStatisticsLogging;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.thirdparty.com.google.common.base.Preconditions;
import org.apache.hadoop.util.DurationInfo;
import org.apache.hadoop.util.ExitUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.LimitedPrivate(value={"management tools"})
@InterfaceStability.Unstable
public final class MarkerTool
extends S3GuardTool {
    private static final Logger LOG = LoggerFactory.getLogger(MarkerTool.class);
    public static final String MARKERS = "markers";
    public static final String PURPOSE = "View and manipulate S3 directory markers";
    public static final String OPT_AUDIT = "audit";
    public static final String OPT_CLEAN = "clean";
    public static final String AUDIT = "-audit";
    public static final String CLEAN = "-clean";
    public static final String OPT_MIN = "min";
    public static final String OPT_MAX = "max";
    public static final String OPT_OUT = "out";
    public static final String OPT_LIMIT = "limit";
    public static final String OPT_NONAUTH = "nonauth";
    @VisibleForTesting
    static final String E_ARGUMENTS = "Wrong number of arguments: %d";
    public static final int UNLIMITED_LISTING = 0;
    public static final int UNLIMITED_MIN_MARKERS = -1;
    private static final String USAGE = "markers (-audit | -clean) [-min <count>] [-max <count>] [-out <filename>] [-limit <limit>] [-nonauth] [-verbose] <PATH>\n\tView and manipulate S3 directory markers\n\n";
    private PrintStream out = System.out;
    private boolean verbose;
    private StoreContext storeContext;
    private MarkerToolOperations operations;

    public MarkerTool(Configuration conf) {
        super(conf, OPT_AUDIT, OPT_CLEAN, "verbose", OPT_NONAUTH);
        CommandFormat format = this.getCommandFormat();
        format.addOptionWithValue(OPT_MIN);
        format.addOptionWithValue(OPT_MAX);
        format.addOptionWithValue(OPT_LIMIT);
        format.addOptionWithValue(OPT_OUT);
    }

    @Override
    public String getUsage() {
        return USAGE;
    }

    @Override
    public String getName() {
        return MARKERS;
    }

    @Override
    public void resetBindings() {
        super.resetBindings();
        this.storeContext = null;
        this.operations = null;
    }

    @Override
    public int run(String[] args, PrintStream stream) throws ExitUtil.ExitException, Exception {
        String saveFile;
        ScanResult result;
        boolean clean;
        List<String> parsedArgs;
        this.out = stream;
        try {
            parsedArgs = this.parseArgs(args);
        }
        catch (CommandFormat.UnknownOptionException e) {
            MarkerTool.errorln(this.getUsage());
            throw new ExitUtil.ExitException(42, e.getMessage(), e);
        }
        if (parsedArgs.size() != 1) {
            MarkerTool.errorln(this.getUsage());
            MarkerTool.println(this.out, "Supplied arguments: [" + String.join((CharSequence)", ", parsedArgs) + "]", new Object[0]);
            throw new ExitUtil.ExitException(42, String.format(E_ARGUMENTS, parsedArgs.size()));
        }
        CommandFormat command = this.getCommandFormat();
        this.verbose = command.getOpt("verbose");
        int expectedMin = this.getOptValue(OPT_MIN, 0);
        int expectedMax = this.getOptValue(OPT_MAX, 0);
        boolean audit = command.getOpt(OPT_AUDIT);
        if (audit == (clean = command.getOpt(OPT_CLEAN))) {
            MarkerTool.errorln(this.getUsage());
            throw new ExitUtil.ExitException(42, "Exactly one of -audit and -clean");
        }
        int limit = this.getOptValue(OPT_LIMIT, 0);
        String dir = parsedArgs.get(0);
        Path path = new Path(dir);
        URI uri = path.toUri();
        if (uri.getPath().isEmpty()) {
            path = new Path(path, "/");
        }
        FileSystem fs = path.getFileSystem(this.getConf());
        boolean nonAuth = command.getOpt(OPT_NONAUTH);
        try {
            result = this.execute(new ScanArgsBuilder().withSourceFS(fs).withPath(path).withDoPurge(clean).withMinMarkerCount(expectedMin).withMaxMarkerCount(expectedMax).withLimit(limit).withNonAuth(nonAuth).build());
        }
        catch (UnknownStoreException ex) {
            throw new ExitUtil.ExitException(44, ex.toString(), ex);
        }
        if (this.verbose) {
            this.dumpFileSystemStatistics(this.out);
        }
        if ((saveFile = command.getOptValue(OPT_OUT)) != null && !saveFile.isEmpty()) {
            MarkerTool.println(this.out, "Saving result to %s", saveFile);
            try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new FileOutputStream(saveFile), StandardCharsets.UTF_8);){
                List surplus = result.getTracker().getSurplusMarkers().keySet().stream().map(p -> p.toString() + "/").sorted().collect(Collectors.toList());
                IOUtils.writeLines(surplus, "\n", writer);
            }
        }
        return result.finish();
    }

    private int getOptValue(String option, int defVal) {
        CommandFormat command = this.getCommandFormat();
        String value = command.getOptValue(option);
        if (value != null && !value.isEmpty()) {
            try {
                return Integer.parseInt(value);
            }
            catch (NumberFormatException e) {
                throw new ExitUtil.ExitException(42, String.format("Argument for %s is not a number: %s", option, value));
            }
        }
        return defVal;
    }

    @VisibleForTesting
    ScanResult execute(ScanArgs scanArgs) throws IOException {
        S3AFileSystem fs = this.bindFilesystem(scanArgs.getSourceFS());
        this.storeContext = fs.createStoreContext();
        DirectoryPolicy activePolicy = fs.getDirectoryMarkerPolicy();
        DirectoryPolicy.MarkerPolicy policy = activePolicy.getMarkerPolicy();
        MarkerTool.println(this.out, "The directory marker policy of %s is \"%s\"", new Object[]{this.storeContext.getFsURI(), policy});
        String authPath = this.storeContext.getConfiguration().getTrimmed("fs.s3a.authoritative.path", "");
        if (policy == DirectoryPolicy.MarkerPolicy.Authoritative) {
            MarkerTool.println(this.out, "Authoritative path list is \"%s\"", authPath);
        }
        Path path = scanArgs.getPath();
        Path target = path.makeQualified(fs.getUri(), new Path("/"));
        try {
            this.getFilesystem().getFileStatus(target);
        }
        catch (UnknownStoreException ex) {
            throw new ExitUtil.ExitException(44, ex.toString(), ex);
        }
        catch (FileNotFoundException ex) {
            throw new ExitUtil.ExitException(44, "Not found: " + target, ex);
        }
        DirectoryPolicyImpl filterPolicy = scanArgs.isNonAuth() ? new DirectoryPolicyImpl(DirectoryPolicy.MarkerPolicy.Authoritative, fs::allowAuthoritative) : null;
        int minMarkerCount = scanArgs.getMinMarkerCount();
        int maxMarkerCount = scanArgs.getMaxMarkerCount();
        if (minMarkerCount > maxMarkerCount) {
            MarkerTool.println(this.out, "Swapping -min (%d) and -max (%d) values", minMarkerCount, maxMarkerCount);
            int m3 = minMarkerCount;
            minMarkerCount = maxMarkerCount;
            maxMarkerCount = m3;
        }
        this.operations = fs.createMarkerToolOperations(target.toString());
        return this.scan(target, scanArgs.isDoPurge(), minMarkerCount, maxMarkerCount, scanArgs.getLimit(), filterPolicy);
    }

    private ScanResult scan(Path path, boolean doPurge, int minMarkerCount, int maxMarkerCount, int limit, DirectoryPolicy filterPolicy) throws IOException, ExitUtil.ExitException {
        boolean completed;
        Preconditions.checkArgument(minMarkerCount <= maxMarkerCount, "The min marker count of %d is greater than the max value of %d", minMarkerCount, maxMarkerCount);
        ScanResult result = new ScanResult();
        result.exitCode = 0;
        DirMarkerTracker tracker = new DirMarkerTracker(path, true);
        result.tracker = tracker;
        try (DurationInfo ignored = new DurationInfo(LOG, "marker scan %s", path);){
            completed = this.scanDirectoryTree(path, tracker, limit);
        }
        int objectsFound = tracker.getObjectsFound();
        MarkerTool.println(this.out, "Listed %d object%s under %s%n", objectsFound, this.suffix(objectsFound), path);
        Map<Path, DirMarkerTracker.Marker> surplusMarkers = tracker.getSurplusMarkers();
        Map<Path, DirMarkerTracker.Marker> leafMarkers = tracker.getLeafMarkers();
        int markerCount = surplusMarkers.size();
        result.totalMarkerCount = markerCount;
        result.filteredMarkerCount = markerCount;
        if (markerCount == 0) {
            MarkerTool.println(this.out, "No surplus directory markers were found under %s", path);
        } else {
            MarkerTool.println(this.out, "Found %d surplus directory marker%s under %s", markerCount, this.suffix(markerCount), path);
            for (Path markers : surplusMarkers.keySet()) {
                MarkerTool.println(this.out, "    %s/", markers);
            }
        }
        if (!leafMarkers.isEmpty()) {
            MarkerTool.println(this.out, "Found %d empty directory 'leaf' marker%s under %s", leafMarkers.size(), this.suffix(leafMarkers.size()), path);
            for (Path markers : leafMarkers.keySet()) {
                MarkerTool.println(this.out, "    %s/", markers);
            }
            MarkerTool.println(this.out, "These are required to indicate empty directories", new Object[0]);
        }
        if (doPurge) {
            int deletePageSize = this.storeContext.getConfiguration().getInt("fs.s3a.bulk.delete.page.size", 250);
            result.purgeSummary = this.purgeMarkers(tracker, deletePageSize);
        } else {
            if (filterPolicy != null) {
                List<Path> allowed = tracker.removeAllowedMarkers(filterPolicy);
                int allowedMarkers = allowed.size();
                MarkerTool.println(this.out, "%nIgnoring %d marker%s in authoritative paths", allowedMarkers, this.suffix(allowedMarkers));
                if (this.verbose) {
                    allowed.forEach(p -> MarkerTool.println(this.out, p.toString(), new Object[0]));
                }
                markerCount = surplusMarkers.size();
                result.filteredMarkerCount = markerCount;
            }
            if (markerCount < minMarkerCount || markerCount > maxMarkerCount) {
                return this.failScan(result, 46, "Marker count %d out of range [%d - %d]", markerCount, minMarkerCount, maxMarkerCount);
            }
        }
        if (!completed) {
            this.failScan(result, 3, "Listing limit (%d) reached before completing the scan", limit);
        }
        return result;
    }

    private ScanResult failScan(ScanResult result, int code, String message, Object ... args) {
        String text = String.format(message, args);
        result.exitCode = code;
        result.exitText = text;
        return result;
    }

    private String suffix(int size) {
        return size == 1 ? "" : "s";
    }

    private boolean scanDirectoryTree(Path path, DirMarkerTracker tracker, int limit) throws IOException {
        int count = 0;
        boolean result = true;
        RemoteIterator<S3AFileStatus> listing = this.operations.listObjects(path, this.storeContext.pathToKey(path));
        while (listing.hasNext()) {
            ++count;
            S3AFileStatus status = listing.next();
            Path statusPath = status.getPath();
            S3ALocatedFileStatus locatedStatus = new S3ALocatedFileStatus(status, null);
            String key = this.storeContext.pathToKey(statusPath);
            if (status.isDirectory()) {
                if (this.verbose) {
                    MarkerTool.println(this.out, "  Directory Marker %s/", key);
                }
                LOG.debug("{}", (Object)key);
                tracker.markerFound(statusPath, key + "/", locatedStatus);
            } else {
                tracker.fileFound(statusPath, key, locatedStatus);
            }
            if (count % 1000 == 0) {
                MarkerTool.println(this.out, "Scanned %,d objects", count);
            }
            if (limit <= 0 || count < limit) continue;
            MarkerTool.println(this.out, "Limit of scan reached - %,d object%s", limit, this.suffix(limit));
            result = false;
            break;
        }
        LOG.debug("Listing summary {}", listing);
        if (this.verbose) {
            MarkerTool.println(this.out, "%nListing statistics:%n  %s%n", IOStatisticsLogging.ioStatisticsSourceToString(listing));
        }
        return result;
    }

    private MarkerPurgeSummary purgeMarkers(DirMarkerTracker tracker, int deletePageSize) throws MultiObjectDeleteException, AmazonClientException, IOException {
        MarkerPurgeSummary summary = new MarkerPurgeSummary();
        Map<Path, DirMarkerTracker.Marker> markers = tracker.getSurplusMarkers();
        int size = markers.size();
        List collect = markers.values().stream().map(p -> new DeleteObjectsRequest.KeyVersion(p.getKey())).collect(Collectors.toList());
        ArrayList markerKeys = new ArrayList(collect);
        Collections.shuffle(markerKeys);
        int pages = size / deletePageSize;
        if (size % deletePageSize > 0) {
            ++pages;
        }
        if (this.verbose) {
            MarkerTool.println(this.out, "%n%d marker%s to delete in %d page%s of %d keys/page", size, this.suffix(size), pages, this.suffix(pages), deletePageSize);
        }
        DurationInfo durationInfo = new DurationInfo(LOG, "Deleting markers", new Object[0]);
        int start = 0;
        while (start < size) {
            int end = Math.min(start + deletePageSize, size);
            List page = markerKeys.subList(start, end);
            ArrayList undeleted = new ArrayList();
            Invoker.once("Remove S3 Keys", tracker.getBasePath().toString(), () -> this.operations.removeKeys(page, true, undeleted, null, false));
            summary.deleteRequests++;
            start = end;
        }
        durationInfo.close();
        summary.totalDeleteRequestDuration = durationInfo.value();
        summary.markersDeleted = size;
        return summary;
    }

    public boolean isVerbose() {
        return this.verbose;
    }

    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    public static ScanResult execMarkerTool(ScanArgs scanArgs) throws IOException {
        MarkerTool tool = new MarkerTool(scanArgs.getSourceFS().getConf());
        tool.setVerbose(LOG.isDebugEnabled());
        return tool.execute(scanArgs);
    }

    public static final class ScanArgsBuilder {
        private FileSystem sourceFS;
        private Path path;
        private boolean doPurge = false;
        private int minMarkerCount = 0;
        private int maxMarkerCount = 0;
        private int limit = 0;
        private boolean nonAuth = false;

        public ScanArgsBuilder withSourceFS(FileSystem source) {
            this.sourceFS = source;
            return this;
        }

        public ScanArgsBuilder withPath(Path p) {
            this.path = p;
            return this;
        }

        public ScanArgsBuilder withDoPurge(boolean d) {
            this.doPurge = d;
            return this;
        }

        public ScanArgsBuilder withMinMarkerCount(int min2) {
            this.minMarkerCount = min2;
            return this;
        }

        public ScanArgsBuilder withMaxMarkerCount(int max) {
            this.maxMarkerCount = max;
            return this;
        }

        public ScanArgsBuilder withLimit(int l) {
            this.limit = l;
            return this;
        }

        public ScanArgsBuilder withNonAuth(boolean b) {
            this.nonAuth = b;
            return this;
        }

        public ScanArgs build() {
            return new ScanArgs(this.sourceFS, this.path, this.doPurge, this.minMarkerCount, this.maxMarkerCount, this.limit, this.nonAuth);
        }
    }

    public static final class ScanArgs {
        private final FileSystem sourceFS;
        private final Path path;
        private final boolean doPurge;
        private final int minMarkerCount;
        private final int maxMarkerCount;
        private final int limit;
        private final boolean nonAuth;

        private ScanArgs(FileSystem sourceFS, Path path, boolean doPurge, int minMarkerCount, int maxMarkerCount, int limit, boolean nonAuth) {
            this.sourceFS = sourceFS;
            this.path = path;
            this.doPurge = doPurge;
            this.minMarkerCount = minMarkerCount;
            this.maxMarkerCount = maxMarkerCount;
            this.limit = limit;
            this.nonAuth = nonAuth;
        }

        FileSystem getSourceFS() {
            return this.sourceFS;
        }

        Path getPath() {
            return this.path;
        }

        boolean isDoPurge() {
            return this.doPurge;
        }

        int getMinMarkerCount() {
            return this.minMarkerCount;
        }

        int getMaxMarkerCount() {
            return this.maxMarkerCount;
        }

        int getLimit() {
            return this.limit;
        }

        boolean isNonAuth() {
            return this.nonAuth;
        }
    }

    public static final class MarkerPurgeSummary {
        private int markersDeleted;
        private int deleteRequests;
        private long totalDeleteRequestDuration;

        public String toString() {
            return "MarkerPurgeSummary{markersDeleted=" + this.markersDeleted + ", deleteRequests=" + this.deleteRequests + ", totalDeleteRequestDuration=" + this.totalDeleteRequestDuration + '}';
        }

        int getMarkersDeleted() {
            return this.markersDeleted;
        }

        int getDeleteRequests() {
            return this.deleteRequests;
        }

        long getTotalDeleteRequestDuration() {
            return this.totalDeleteRequestDuration;
        }
    }

    public static final class ScanResult {
        private int exitCode;
        private String exitText = "";
        private int totalMarkerCount;
        private int filteredMarkerCount;
        private DirMarkerTracker tracker;
        private MarkerPurgeSummary purgeSummary;

        private ScanResult() {
        }

        public String toString() {
            return "ScanResult{exitCode=" + this.exitCode + ", exitText=" + this.exitText + ", totalMarkerCount=" + this.totalMarkerCount + ", filteredMarkerCount=" + this.filteredMarkerCount + ", tracker=" + this.tracker + ", purgeSummary=" + this.purgeSummary + '}';
        }

        public int getExitCode() {
            return this.exitCode;
        }

        public DirMarkerTracker getTracker() {
            return this.tracker;
        }

        public MarkerPurgeSummary getPurgeSummary() {
            return this.purgeSummary;
        }

        public int getTotalMarkerCount() {
            return this.totalMarkerCount;
        }

        public int getFilteredMarkerCount() {
            return this.filteredMarkerCount;
        }

        public int finish() throws ExitUtil.ExitException {
            if (this.exitCode != 0) {
                throw new ExitUtil.ExitException(this.exitCode, this.exitText);
            }
            return 0;
        }
    }
}

