/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.util;

import java.io.PrintStream;
import java.lang.reflect.Constructor;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.Random;
import java.util.zip.CRC32;
import java.util.zip.Checksum;
import org.apache.hadoop.fs.ChecksumException;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.util.DataChecksum;
import org.apache.hadoop.util.NativeCodeLoader;
import org.apache.hadoop.util.NativeCrc32;
import org.apache.hadoop.util.PureJavaCrc32;
import org.apache.hadoop.util.PureJavaCrc32C;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

public class Crc32PerformanceTest {
    static final int MB = 0x100000;
    final int dataLengthMB;
    final int trials;
    final boolean direct;
    final PrintStream out = System.out;
    final List<Class<? extends Crc32>> crcs = new ArrayList<Class<? extends Crc32>>();

    Crc32PerformanceTest(int dataLengthMB, int trials, boolean direct) {
        this.dataLengthMB = dataLengthMB;
        this.trials = trials;
        this.direct = direct;
        this.crcs.add(Crc32.Zip.class);
        this.crcs.add(Crc32.PureJava.class);
        this.crcs.add(Crc32.PureJavaC.class);
        if (NativeCrc32.isAvailable()) {
            if (direct) {
                this.crcs.add(Crc32.Native.class);
            }
            this.crcs.add(Crc32.NativeC.class);
            GenericTestUtils.setLogLevel(LoggerFactory.getLogger(NativeCodeLoader.class), Level.TRACE);
        }
    }

    void run() throws Exception {
        long startTime = System.nanoTime();
        Crc32PerformanceTest.printSystemProperties(this.out);
        this.out.println("Data Length = " + this.dataLengthMB + " MB");
        this.out.println("Trials      = " + this.trials);
        this.doBench(this.crcs);
        this.out.printf("Elapsed %.1fs\n", Crc32PerformanceTest.secondsElapsed(startTime));
    }

    public static void main(String[] args) throws Exception {
        boolean isdirect = true;
        if (args.length > 0) {
            isdirect = false;
        }
        new Crc32PerformanceTest(64, 5, isdirect).run();
    }

    private static void printCell(String s, int width, PrintStream outCrc) {
        int w = s.length() > width ? s.length() : width;
        outCrc.printf(" %" + w + "s |", s);
    }

    private ByteBuffer allocateByteBuffer(int length) {
        return this.direct ? ByteBuffer.allocateDirect(length) : ByteBuffer.allocate(length);
    }

    private ByteBuffer newData() {
        byte[] bytes = new byte[this.dataLengthMB << 20];
        new Random().nextBytes(bytes);
        ByteBuffer dataBufs = this.allocateByteBuffer(bytes.length);
        dataBufs.mark();
        dataBufs.put(bytes);
        dataBufs.reset();
        return dataBufs;
    }

    private ByteBuffer computeCrc(ByteBuffer dataBufs, int bytePerCrc, DataChecksum.Type type) {
        int size = 4 * (dataBufs.remaining() - 1) / bytePerCrc + 1;
        ByteBuffer crcBufs = this.allocateByteBuffer(size);
        DataChecksum checksum = DataChecksum.newDataChecksum((DataChecksum.Type)type, (int)bytePerCrc);
        checksum.calculateChunkedSums(dataBufs, crcBufs);
        return crcBufs;
    }

    private ByteBuffer computeCrc(Class<? extends Crc32> clazz, ByteBuffer dataBufs, int bytePerCrc) throws Exception {
        Constructor<? extends Crc32> ctor = clazz.getConstructor(new Class[0]);
        Crc32 crc = ctor.newInstance(new Object[0]);
        int size = 4 * (dataBufs.remaining() - 1) / bytePerCrc + 1;
        ByteBuffer crcBufs = this.allocateByteBuffer(size);
        DataChecksum checksum = DataChecksum.newDataChecksum((DataChecksum.Type)crc.crcType(), (int)bytePerCrc);
        checksum.calculateChunkedSums(dataBufs, crcBufs);
        return crcBufs;
    }

    private void doBench(List<Class<? extends Crc32>> crcTargets) throws Exception {
        ByteBuffer[] dataBufs = new ByteBuffer[16];
        for (int i = 0; i < dataBufs.length; ++i) {
            dataBufs[i] = this.newData();
        }
        this.out.printf("\n%s Buffer Performance Table", this.direct ? "Direct" : "Non-direct");
        this.out.printf(" (bpc: byte-per-crc in MB/sec; #T: #Theads)\n", new Object[0]);
        for (Class<? extends Crc32> c : crcTargets) {
            ByteBuffer[] crc32 = new ByteBuffer[]{this.computeCrc(c, dataBufs[0], 32)};
            ByteBuffer[] crc512 = new ByteBuffer[]{this.computeCrc(c, dataBufs[0], 512)};
            this.doBench(c, 1, dataBufs, crc32, 32);
            this.doBench(c, 1, dataBufs, crc512, 512);
        }
        for (int i = 5; i <= 16; ++i) {
            this.doBench(this.crcs, dataBufs, 1 << i, this.out);
        }
    }

    private void doBench(List<Class<? extends Crc32>> crcTargets, ByteBuffer[] dataBufs, int bytePerCrc, PrintStream outCrc) throws Exception {
        ByteBuffer[] crcBufs = new ByteBuffer[dataBufs.length];
        ByteBuffer[] crcBufsC = new ByteBuffer[dataBufs.length];
        for (int i = 0; i < dataBufs.length; ++i) {
            crcBufs[i] = this.computeCrc(dataBufs[i], bytePerCrc, DataChecksum.Type.CRC32);
            crcBufsC[i] = this.computeCrc(dataBufs[i], bytePerCrc, DataChecksum.Type.CRC32C);
        }
        String numBytesStr = " bpc ";
        String numThreadsStr = "#T";
        String diffStr = "% diff";
        outCrc.print('|');
        Crc32PerformanceTest.printCell(" bpc ", 0, outCrc);
        Crc32PerformanceTest.printCell("#T", 0, outCrc);
        for (int i = 0; i < crcTargets.size(); ++i) {
            Class<? extends Crc32> c = crcTargets.get(i);
            outCrc.print('|');
            Crc32PerformanceTest.printCell(c.getSimpleName(), 8, outCrc);
            if (i <= 0) continue;
            Crc32PerformanceTest.printCell("% diff", "% diff".length(), outCrc);
        }
        outCrc.printf("\n", new Object[0]);
        for (int numThreads = 1; numThreads <= dataBufs.length; numThreads <<= 1) {
            outCrc.printf("|", new Object[0]);
            Crc32PerformanceTest.printCell(String.valueOf(bytePerCrc), " bpc ".length(), outCrc);
            Crc32PerformanceTest.printCell(String.valueOf(numThreads), "#T".length(), outCrc);
            ArrayList<BenchResult> previous = new ArrayList<BenchResult>();
            for (Class<? extends Crc32> c : crcTargets) {
                System.gc();
                Constructor<? extends Crc32> ctor = c.getConstructor(new Class[0]);
                Crc32 crc = ctor.newInstance(new Object[0]);
                BenchResult result = crc.crcType() == DataChecksum.Type.CRC32 ? this.doBench(c, numThreads, dataBufs, crcBufs, bytePerCrc) : this.doBench(c, numThreads, dataBufs, crcBufsC, bytePerCrc);
                Crc32PerformanceTest.printCell(String.format("%9.1f", result.mbps), c.getSimpleName().length() + 1, outCrc);
                int size = previous.size();
                if (size > 0) {
                    BenchResult p = (BenchResult)previous.get(size - 1);
                    double diff = (result.mbps - p.mbps) / p.mbps * 100.0;
                    Crc32PerformanceTest.printCell(String.format("%5.1f%%", diff), "% diff".length(), outCrc);
                }
                previous.add(result);
            }
            outCrc.printf("\n", new Object[0]);
        }
    }

    private BenchResult doBench(Class<? extends Crc32> clazz, int numThreads, final ByteBuffer[] dataBufs, final ByteBuffer[] crcBufs, final int bytePerCrc) throws Exception {
        Thread[] threads = new Thread[numThreads];
        final BenchResult[] results = new BenchResult[threads.length];
        Constructor<? extends Crc32> ctor = clazz.getConstructor(new Class[0]);
        for (int i = 0; i < threads.length; ++i) {
            final Crc32 crc = ctor.newInstance(new Object[0]);
            final long byteProcessed = dataBufs[i].remaining() * this.trials;
            final int index = i;
            threads[i] = new Thread(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void run() {
                    long startTime = System.nanoTime();
                    for (int i = 0; i < Crc32PerformanceTest.this.trials; ++i) {
                        dataBufs[index].mark();
                        crcBufs[index].mark();
                        try {
                            crc.verifyChunked(dataBufs[index], bytePerCrc, crcBufs[index], crc.getClass().getSimpleName(), dataBufs[index].position());
                            continue;
                        }
                        catch (Throwable t) {
                            results[index] = new BenchResult(t);
                            return;
                        }
                        finally {
                            dataBufs[index].reset();
                            crcBufs[index].reset();
                        }
                    }
                    double secsElapsed = Crc32PerformanceTest.secondsElapsed(startTime);
                    results[index] = new BenchResult((double)byteProcessed / secsElapsed / 1048576.0);
                }
            };
        }
        for (Thread t : threads) {
            t.start();
        }
        for (Thread t : threads) {
            t.join();
        }
        double sum = 0.0;
        for (int i = 0; i < results.length; ++i) {
            sum += results[i].getMbps();
        }
        return new BenchResult(sum / (double)results.length);
    }

    static double secondsElapsed(long startTime) {
        return (double)(System.nanoTime() - startTime) / 1.0E9;
    }

    static void printSystemProperties(PrintStream outCrc) {
        String[] names = new String[]{"java.version", "java.runtime.name", "java.runtime.version", "java.vm.version", "java.vm.vendor", "java.vm.name", "java.vm.specification.version", "java.specification.version", "os.arch", "os.name", "os.version"};
        int max = 0;
        for (String n : names) {
            if (n.length() <= max) continue;
            max = n.length();
        }
        Properties p = System.getProperties();
        for (String n : names) {
            outCrc.printf("%" + max + "s = %s\n", n, p.getProperty(n));
        }
    }

    private static class BenchResult {
        final double mbps;
        final Throwable thrown;

        BenchResult(double mbps) {
            this.mbps = mbps;
            this.thrown = null;
        }

        BenchResult(Throwable e) {
            this.mbps = 0.0;
            this.thrown = e;
        }

        double getMbps() {
            if (this.thrown != null) {
                throw new AssertionError((Object)this.thrown);
            }
            return this.mbps;
        }
    }

    static interface Crc32 {
        public void verifyChunked(ByteBuffer var1, int var2, ByteBuffer var3, String var4, long var5) throws ChecksumException;

        public DataChecksum.Type crcType();

        public static final class PureJavaC
        extends AbstractCrc32<PureJavaCrc32C> {
            @Override
            public PureJavaCrc32C newAlgorithm() {
                return new PureJavaCrc32C();
            }

            @Override
            public DataChecksum.Type crcType() {
                return DataChecksum.Type.CRC32C;
            }
        }

        public static final class PureJava
        extends AbstractCrc32<PureJavaCrc32> {
            @Override
            public PureJavaCrc32 newAlgorithm() {
                return new PureJavaCrc32();
            }

            @Override
            public DataChecksum.Type crcType() {
                return DataChecksum.Type.CRC32;
            }
        }

        public static final class Zip
        extends AbstractCrc32<CRC32> {
            @Override
            public CRC32 newAlgorithm() {
                return new CRC32();
            }

            @Override
            public DataChecksum.Type crcType() {
                return DataChecksum.Type.CRC32;
            }
        }

        public static abstract class AbstractCrc32<T extends Checksum>
        implements Crc32 {
            abstract T newAlgorithm();

            @Override
            public void verifyChunked(ByteBuffer data, int bytesPerCrc, ByteBuffer sums, String filename, long basePos) throws ChecksumException {
                T algorithm = this.newAlgorithm();
                DataChecksum.Type type = this.crcType();
                if (data.hasArray() && sums.hasArray()) {
                    DataChecksum.verifyChunked((DataChecksum.Type)type, algorithm, (byte[])data.array(), (int)data.position(), (int)data.remaining(), (int)bytesPerCrc, (byte[])sums.array(), (int)sums.position(), (String)filename, (long)basePos);
                } else {
                    DataChecksum.verifyChunked((DataChecksum.Type)type, algorithm, (ByteBuffer)data, (int)bytesPerCrc, (ByteBuffer)sums, (String)filename, (long)basePos);
                }
            }
        }

        public static final class NativeC
        implements Crc32 {
            @Override
            public void verifyChunked(ByteBuffer data, int bytesPerSum, ByteBuffer sums, String fileName, long basePos) throws ChecksumException {
                if (data.isDirect()) {
                    NativeCrc32.verifyChunkedSums((int)bytesPerSum, (int)DataChecksum.Type.CRC32C.id, (ByteBuffer)sums, (ByteBuffer)data, (String)fileName, (long)basePos);
                } else {
                    int dataOffset = data.arrayOffset() + data.position();
                    int crcsOffset = sums.arrayOffset() + sums.position();
                    NativeCrc32.verifyChunkedSumsByteArray((int)bytesPerSum, (int)DataChecksum.Type.CRC32C.id, (byte[])sums.array(), (int)crcsOffset, (byte[])data.array(), (int)dataOffset, (int)data.remaining(), (String)fileName, (long)basePos);
                }
            }

            @Override
            public DataChecksum.Type crcType() {
                return DataChecksum.Type.CRC32C;
            }
        }

        public static final class Native
        implements Crc32 {
            @Override
            public void verifyChunked(ByteBuffer data, int bytesPerSum, ByteBuffer sums, String fileName, long basePos) throws ChecksumException {
                NativeCrc32.verifyChunkedSums((int)bytesPerSum, (int)DataChecksum.Type.CRC32.id, (ByteBuffer)sums, (ByteBuffer)data, (String)fileName, (long)basePos);
            }

            @Override
            public DataChecksum.Type crcType() {
                return DataChecksum.Type.CRC32;
            }
        }
    }
}

