/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.connect.runtime.errors;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.errors.RetriableException;
import org.apache.kafka.connect.runtime.errors.ErrorHandlingMetrics;
import org.apache.kafka.connect.runtime.errors.ErrorReporter;
import org.apache.kafka.connect.runtime.errors.Operation;
import org.apache.kafka.connect.runtime.errors.ProcessingContext;
import org.apache.kafka.connect.runtime.errors.Stage;
import org.apache.kafka.connect.runtime.errors.ToleranceType;
import org.apache.kafka.connect.runtime.errors.WorkerErrantRecordReporter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RetryWithToleranceOperator<T>
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(RetryWithToleranceOperator.class);
    public static final long RETRIES_DELAY_MIN_MS = 300L;
    private static final Map<Stage, Class<? extends Exception>> TOLERABLE_EXCEPTIONS = new HashMap<Stage, Class<? extends Exception>>();
    private final long errorRetryTimeout;
    private final long errorMaxDelayInMillis;
    private final ToleranceType errorToleranceType;
    private long totalFailures = 0L;
    private final Time time;
    private final ErrorHandlingMetrics errorHandlingMetrics;
    private final CountDownLatch stopRequestedLatch;
    private volatile boolean stopping;
    private List<ErrorReporter<T>> reporters;

    public RetryWithToleranceOperator(long errorRetryTimeout, long errorMaxDelayInMillis, ToleranceType toleranceType, Time time, ErrorHandlingMetrics errorHandlingMetrics) {
        this(errorRetryTimeout, errorMaxDelayInMillis, toleranceType, time, errorHandlingMetrics, new CountDownLatch(1));
    }

    RetryWithToleranceOperator(long errorRetryTimeout, long errorMaxDelayInMillis, ToleranceType toleranceType, Time time, ErrorHandlingMetrics errorHandlingMetrics, CountDownLatch stopRequestedLatch) {
        this.errorRetryTimeout = errorRetryTimeout;
        this.errorMaxDelayInMillis = errorMaxDelayInMillis;
        this.errorToleranceType = toleranceType;
        this.time = time;
        this.errorHandlingMetrics = errorHandlingMetrics;
        this.stopRequestedLatch = stopRequestedLatch;
        this.stopping = false;
        this.reporters = List.of();
    }

    public Future<Void> executeFailed(ProcessingContext<T> context, Stage stage, Class<?> executingClass, Throwable error) {
        this.markAsFailed();
        context.currentContext(stage, executingClass);
        context.error(error);
        this.errorHandlingMetrics.recordFailure();
        Future<Void> errantRecordFuture = this.report(context);
        if (!this.withinToleranceLimits()) {
            this.errorHandlingMetrics.recordError();
            throw new ConnectException("Tolerance exceeded in error handler", error);
        }
        return errantRecordFuture;
    }

    synchronized Future<Void> report(ProcessingContext<T> context) {
        if (this.reporters.size() == 1) {
            return new WorkerErrantRecordReporter.ErrantRecordFuture(List.of(this.reporters.get(0).report(context)));
        }
        List<Future<RecordMetadata>> futures = this.reporters.stream().map(r -> r.report(context)).filter(f -> !f.isDone()).collect(Collectors.toList());
        if (futures.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        return new WorkerErrantRecordReporter.ErrantRecordFuture(futures);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public <V> V execute(ProcessingContext<T> context, Operation<V> operation, Stage stage, Class<?> executingClass) {
        context.currentContext(stage, executingClass);
        if (context.failed()) {
            log.debug("ProcessingContext is already in failed state. Ignoring requested operation.");
            return null;
        }
        context.currentContext(stage, executingClass);
        try {
            Class<RetriableException> ex = TOLERABLE_EXCEPTIONS.getOrDefault((Object)context.stage(), RetriableException.class);
            V v = this.execAndHandleError(context, operation, ex);
            return v;
        }
        finally {
            if (context.failed()) {
                this.errorHandlingMetrics.recordError();
                this.report(context);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected <V> V execAndRetry(ProcessingContext<T> context, Operation<V> operation) throws Exception {
        int attempt = 0;
        long startTime = this.time.milliseconds();
        long deadline = this.errorRetryTimeout >= 0L ? startTime + this.errorRetryTimeout : Long.MAX_VALUE;
        while (true) {
            try {
                ++attempt;
                Object v = operation.call();
                return v;
            }
            catch (RetriableException e) {
                log.trace("Caught a retriable exception while executing {} operation with {}", (Object)context.stage(), context.executingClass());
                this.errorHandlingMetrics.recordFailure();
                if (this.time.milliseconds() >= deadline) {
                    log.trace("Can't retry. start={}, attempt={}, deadline={}", new Object[]{startTime, attempt, deadline});
                    throw e;
                }
                this.backoff(attempt, deadline);
                this.errorHandlingMetrics.recordRetry();
                if (!this.stopping) continue;
                log.trace("Shutdown has been scheduled. Marking operation as failed.");
                context.error(e);
                V v = null;
                return v;
            }
            finally {
                context.attempt(attempt);
                continue;
            }
            break;
        }
    }

    protected <V> V execAndHandleError(ProcessingContext<T> context, Operation<V> operation, Class<? extends Exception> tolerated) {
        try {
            V result = this.execAndRetry(context, operation);
            if (context.failed()) {
                this.markAsFailed();
                this.errorHandlingMetrics.recordSkipped();
            }
            return result;
        }
        catch (Exception e) {
            this.errorHandlingMetrics.recordFailure();
            this.markAsFailed();
            context.error(e);
            if (!tolerated.isAssignableFrom(e.getClass())) {
                throw new ConnectException("Unhandled exception in error handler", (Throwable)e);
            }
            if (!this.withinToleranceLimits()) {
                throw new ConnectException("Tolerance exceeded in error handler", (Throwable)e);
            }
            this.errorHandlingMetrics.recordSkipped();
            return null;
        }
    }

    synchronized void markAsFailed() {
        this.errorHandlingMetrics.recordErrorTimestamp();
        ++this.totalFailures;
    }

    public synchronized boolean withinToleranceLimits() {
        switch (this.errorToleranceType) {
            case NONE: {
                if (this.totalFailures > 0L) {
                    return false;
                }
            }
            case ALL: {
                return true;
            }
        }
        throw new ConfigException("Unknown tolerance type: {}", (Object)this.errorToleranceType);
    }

    public ToleranceType getErrorToleranceType() {
        return this.errorToleranceType;
    }

    void backoff(int attempt, long deadline) {
        long currentTime;
        int numRetry = attempt - 1;
        long delay = 300L << numRetry;
        if (delay > this.errorMaxDelayInMillis) {
            delay = ThreadLocalRandom.current().nextLong(this.errorMaxDelayInMillis);
        }
        if (delay + (currentTime = this.time.milliseconds()) > deadline) {
            delay = Math.max(0L, deadline - currentTime);
        }
        log.debug("Sleeping for up to {} millis", (Object)delay);
        try {
            this.stopRequestedLatch.await(delay, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public String toString() {
        return "RetryWithToleranceOperator{errorRetryTimeout=" + this.errorRetryTimeout + ", errorMaxDelayInMillis=" + this.errorMaxDelayInMillis + ", errorToleranceType=" + String.valueOf((Object)this.errorToleranceType) + ", totalFailures=" + this.totalFailures + ", time=" + String.valueOf(this.time) + "}";
    }

    public synchronized void reporters(List<ErrorReporter<T>> reporters) {
        this.reporters = Objects.requireNonNull(reporters, "reporters");
    }

    public void triggerStop() {
        this.stopping = true;
        this.stopRequestedLatch.countDown();
    }

    @Override
    public synchronized void close() {
        ConnectException e = null;
        for (ErrorReporter<T> reporter : this.reporters) {
            try {
                reporter.close();
            }
            catch (Throwable t) {
                e = e != null ? e : new ConnectException("Failed to close all reporters");
                e.addSuppressed(t);
            }
        }
        this.reporters = List.of();
        if (e != null) {
            throw e;
        }
    }

    static {
        TOLERABLE_EXCEPTIONS.put(Stage.TRANSFORMATION, Exception.class);
        TOLERABLE_EXCEPTIONS.put(Stage.HEADER_CONVERTER, Exception.class);
        TOLERABLE_EXCEPTIONS.put(Stage.KEY_CONVERTER, Exception.class);
        TOLERABLE_EXCEPTIONS.put(Stage.VALUE_CONVERTER, Exception.class);
    }
}

