/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

#ifndef TVM_META_SCHEDULE_COST_MODEL_H_
#define TVM_META_SCHEDULE_COST_MODEL_H_

#include <tvm/ffi/container/array.h>
#include <tvm/ffi/function.h>
#include <tvm/ffi/reflection/registry.h>
#include <tvm/ffi/string.h>
#include <tvm/meta_schedule/arg_info.h>
#include <tvm/meta_schedule/measure_candidate.h>
#include <tvm/meta_schedule/runner.h>
#include <tvm/runtime/object.h>
#include <tvm/tir/schedule/schedule.h>

#include <vector>

namespace tvm {
namespace meta_schedule {

class TuneContext;

/*! \brief Cost model. */
class CostModelNode : public runtime::Object {
 public:
  /*! \brief Virtual destructor. */
  virtual ~CostModelNode() = default;

  /*!
   * \brief Load the cost model from given file location.
   * \param path The file path.
   */
  virtual void Load(const ffi::String& path) = 0;

  /*!
   * \brief Save the cost model to given file location.
   * \param path The file path.
   */
  virtual void Save(const ffi::String& path) = 0;

  /*!
   * \brief Update the cost model given running results.
   * \param context The tuning context.
   * \param candidates The measure candidates.
   * \param results The running results of the measure candidates.
   */
  virtual void Update(const TuneContext& context, const ffi::Array<MeasureCandidate>& candidates,
                      const ffi::Array<RunnerResult>& results) = 0;

  /*!
   * \brief Predict the normalized score (the larger the better) of given measure candidates.
   * \param context The tuning context.
   * \param candidates The measure candidates.
   * \return The predicted normalized score.
   */
  virtual std::vector<double> Predict(const TuneContext& context,
                                      const ffi::Array<MeasureCandidate>& candidates) = 0;

  static constexpr const bool _type_mutable = true;
  TVM_FFI_DECLARE_OBJECT_INFO("meta_schedule.CostModel", CostModelNode, Object);
};

/*! \brief The cost model with customized methods on the python-side. */
class PyCostModelNode : public CostModelNode {
 public:
  /*!
   * \brief Load the cost model from given file location.
   * \param path The file path.
   */
  using FLoad = ffi::TypedFunction<void(ffi::String)>;
  /*!
   * \brief Save the cost model to given file location.
   * \param path The file path.
   */
  using FSave = ffi::TypedFunction<void(ffi::String)>;
  /*!
   * \brief Update the cost model given running results.
   * \param context The tuning context.
   * \param candidates The measure candidates.
   * \param results The running results of the measure candidates.
   * \return Whether cost model was updated successfully.
   */
  using FUpdate = ffi::TypedFunction<void(const TuneContext&, const ffi::Array<MeasureCandidate>&,
                                          const ffi::Array<RunnerResult>&)>;
  /*!
   * \brief Predict the running results of given measure candidates.
   * \param context The tuning context.
   * \param candidates The measure candidates.
   * \param p_addr The address to save the estimated running results.
   */
  using FPredict = ffi::TypedFunction<void(const TuneContext&, const ffi::Array<MeasureCandidate>&,
                                           void* p_addr)>;
  /*!
   * \brief Get the cost model as string with name.
   * \return The string representation of the cost model.
   */
  using FAsString = ffi::TypedFunction<ffi::String()>;

  /*! \brief The packed function to the `Load` function. */
  FLoad f_load;
  /*! \brief The packed function to the `Save` function. */
  FSave f_save;
  /*! \brief The packed function to the `Update` function. */
  FUpdate f_update;
  /*! \brief The packed function to the `Predict` function. */
  FPredict f_predict;
  /*! \brief The packed function to the `AsString` function. */
  FAsString f_as_string;

  void Load(const ffi::String& path);
  void Save(const ffi::String& path);
  void Update(const TuneContext& context, const ffi::Array<MeasureCandidate>& candidates,
              const ffi::Array<RunnerResult>& results);
  std::vector<double> Predict(const TuneContext& context,
                              const ffi::Array<MeasureCandidate>& candidates);
  TVM_FFI_DECLARE_OBJECT_INFO_FINAL("meta_schedule.PyCostModel", PyCostModelNode, CostModelNode);
};

/*!
 * \brief Managed reference to CostModelNode
 * \sa CostModelNode
 */
class CostModel : public runtime::ObjectRef {
 public:
  /*!
   * \brief Create a feature extractor with customized methods on the python-side.
   * \param f_load The packed function of `Load`.
   * \param f_save The packed function of `Save`.
   * \param f_update The packed function of `Update`.
   * \param f_predict The packed function of `Predict`.
   * \param f_as_string The packed function of `AsString`.
   * \return The feature extractor created.
   */
  TVM_DLL static CostModel PyCostModel(PyCostModelNode::FLoad f_load,        //
                                       PyCostModelNode::FSave f_save,        //
                                       PyCostModelNode::FUpdate f_update,    //
                                       PyCostModelNode::FPredict f_predict,  //
                                       PyCostModelNode::FAsString f_as_string);
  TVM_FFI_DEFINE_OBJECT_REF_METHODS_NULLABLE(CostModel, ObjectRef, CostModelNode);
};

}  // namespace meta_schedule
}  // namespace tvm

#endif  // TVM_META_SCHEDULE_COST_MODEL_H_
